package gov.cms.grouper.snf.component.v100.logic;

import gov.cms.grouper.snf.component.v100.TestUtil;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.AssessmentType;
import gov.cms.grouper.snf.model.ClinicalCategory;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class TherapyLogicTest {
  public static final int version = 100;

  @Test
  public void testGetPdxClinicalCategory() {
    List<Assessment> assessments = new ArrayList<>(Arrays
        .asList(new Assessment(Rai300.J2100.name(), Rai300.J2100.name(), Assessment.CHECKED)));
    assessments.addAll(TestUtil.getAll(Rai300.getAssessmentsByType(AssessmentType.PROCEDURE_MAJOR),
        Assessment.CHECKED));
    ClaimInfo claim = new ClaimInfo(version, false, assessments);
    ClinicalCategory expected = ClinicalCategory.MAJOR_JOINT_REPLACEMENT_OR_SPINAL_SURGERY;
    ClinicalCategory actual = TherapyLogic.getPdxClinicalCategory(version, claim, "M80011S");
    Assertions.assertEquals(expected, actual);

    assessments = new ArrayList<>();
    claim = new ClaimInfo(version, false, assessments);
    expected = ClinicalCategory.ACUTE_INFECTIONS;
    actual = TherapyLogic.getPdxClinicalCategory(version, claim, "N10");
    Assertions.assertEquals(expected, actual);

    assessments = new ArrayList<>(Arrays
        .asList(new Assessment(Rai300.J2100.name(), Rai300.J2100.name(), Assessment.CHECKED)));
    assessments.addAll(TestUtil.getAll(
        Rai300.getAssessmentsByType(AssessmentType.PROCEDURE_ORTHOPEDIC), Assessment.CHECKED));
    claim = new ClaimInfo(version, false, assessments);
    expected = ClinicalCategory.ORTHOPEDIC_SURGERY;
    actual = TherapyLogic.getPdxClinicalCategory(version, claim, "M80011S");
    Assertions.assertEquals(expected, actual);


    assessments = new ArrayList<>(Arrays
        .asList(new Assessment(Rai300.J2100.name(), Rai300.J2100.name(), Assessment.CHECKED)));
    assessments.addAll(TestUtil.getAll(
        Rai300.getAssessmentsByType(AssessmentType.PROCEDURE_NON_ORTHOPEDIC), Assessment.CHECKED));
    claim = new ClaimInfo(version, false, assessments);
    expected = ClinicalCategory.NON_ORTHOPEDIC_SURGERY;
    actual = TherapyLogic.getPdxClinicalCategory(100, claim, "I2602");
    Assertions.assertEquals(expected, actual);

    assessments = new ArrayList<>(Arrays
        .asList(new Assessment(Rai300.J2100.name(), Rai300.J2100.name(), Assessment.CHECKED)));
    claim = new ClaimInfo(version, false, assessments);
    expected = ClinicalCategory.NON_SURGICAL_ORTHOPEDIC_MUSCULOSKELETAL;
    actual = TherapyLogic.getPdxClinicalCategory(101, claim, "M80011P");
    Assertions.assertEquals(expected, actual);

    // invalid pdx
    try {
      Assertions.assertNull(TherapyLogic.getPdxClinicalCategory(100, claim, ""));
    } catch (RuntimeException ex) {

    }
    try {
      // wrong version number
      Assertions.assertNull(TherapyLogic.getPdxClinicalCategory(100, claim, "U071"));
    } catch (RuntimeException ex) {
    }
    try {
      Assertions.assertNull(TherapyLogic.getPdxClinicalCategory(100, claim, "A02020202020"));
    } catch (RuntimeException ex) {

    }
  }


  @Test
  public void testStep2() {
    Assertions.assertEquals(ClinicalCategory.MEDICAL_MANAGEMENT,
        TherapyLogic.step2(100, ClinicalCategory.ACUTE_INFECTIONS));
    Assertions.assertEquals(ClinicalCategory.ACUTE_NEUROLOGIC,
        TherapyLogic.step2(100, ClinicalCategory.ACUTE_NEUROLOGIC));
    Assertions.assertEquals(ClinicalCategory.OTHER_ORTHOPEDIC,
        TherapyLogic.step2(100, ClinicalCategory.ORTHOPEDIC_SURGERY));
  }

  @Test
  public void testStep4() {
    Assertions.assertEquals("TB",
        TherapyLogic
            .step4(100, () -> ClinicalCategory.MAJOR_JOINT_REPLACEMENT_OR_SPINAL_SURGERY, () -> 8)
            .getCmg());
    Assertions.assertEquals("TB",
        TherapyLogic
            .step4(100, () -> ClinicalCategory.MAJOR_JOINT_REPLACEMENT_OR_SPINAL_SURGERY, () -> 9)
            .getCmg());
    Assertions.assertEquals("TK",
        TherapyLogic.step4(100, () -> ClinicalCategory.MEDICAL_MANAGEMENT, () -> 10).getCmg());
    Assertions.assertEquals("TN",
        TherapyLogic.step4(100, () -> ClinicalCategory.NON_ORTHOPEDIC_SURGERY, () -> 6).getCmg());
    Assertions.assertEquals("TP",
        TherapyLogic.step4(100, () -> ClinicalCategory.NON_ORTHOPEDIC_SURGERY, () -> 24).getCmg());

  }

  @Test
  public void testExec() {
    List<Assessment> assessments = Arrays.asList(new Assessment("GG0170B5", "GG0170B5", 5),
        new Assessment("GG0170C5", "GG0170C5", Assessment.CHECKED),
        new Assessment("GG0170D5", "GG0170D5", 0), new Assessment("GG0170E5", "GG0170E5", 0),
        new Assessment("GG0170F5", "GG0170F5", 0), new Assessment("GG0170J5", "GG0170J5", 0),
        new Assessment("GG0170K5", "GG0170K5", 0), new Assessment("GG0130A5", "GG0130A5", 3),
        new Assessment("GG0130B5", "GG0130B5", 3), new Assessment("GG0130C5", "GG0130C5", 4));
    ClaimInfo claim = ClaimInfo.of(version, true, assessments);
    TherapyLogic com = new TherapyLogic(100, claim, ClinicalCategory.ACUTE_INFECTIONS);

    Assertions.assertEquals("TJ", com.exec());

    claim = ClaimInfo.of(version, false, assessments);
    com = new TherapyLogic(100, claim, ClinicalCategory.ACUTE_INFECTIONS);
    String actual = com.exec();
    String expected = "TI";
    Assertions.assertEquals(expected, actual);
  }
}
